/**
 * \file TrkLevel1.h
 * \author Elena Vannuccini
 */
#ifndef trklevel1_h
#define trklevel1_h

#include <TObject.h>
#include <TClonesArray.h>
#include <TRef.h>

#include <TrkStruct.h>

#define INC_CUT 4.
/**
 * \brief Class to describe tracker clusters. 
 *
 * A cluster is defined as a group of adjacent strips whose signals exceed noise by INC_CUT sigma, and at least one strip has signal/noise > SEED_CUT. Tipical values are SEED_CUT = 7 and INC_CUT = 4.
 * 
 */
// ==================================================================
class TrkCluster : public TObject {

private:


public:
	
    int view; ///< view
    int maxs; ///< strip number (1-3072) of cluster center
    int indmax; 
    
    Int_t         CLlength; ///< number of stored strip info (signal+sigma+adc+bad)
    Float_t      *clsignal; //[CLlength] 
    Float_t      *clsigma;  //[CLlength]
    Int_t        *cladc;    //[CLlength]
    Bool_t       *clbad;    //[CLlength]
    
    TrkCluster();
    ~TrkCluster();
    TrkCluster(const TrkCluster&);

    void Dump();
	
	
    Float_t GetSignal(Int_t, Float_t);    ///< cluster signal
    Float_t GetSignal(Float_t cut)  { return GetSignal( (Int_t)0, cut); };
    Float_t GetSignal(Int_t nstrip) { return GetSignal( nstrip, (Float_t)(-1000) ); };
    Float_t GetSignal()             { return GetSignal( (Int_t)0,(Float_t)INC_CUT); };
    
    Float_t GetSignalToNoise(Int_t,Float_t);  ///< cluster signal/noise
    Float_t GetSignalToNoise(Float_t cut)  { return GetSignalToNoise( (Int_t)0, cut); };
    Float_t GetSignalToNoise(Int_t nstrip) { return GetSignalToNoise( nstrip, (Float_t)(-1000) ); };
    Float_t GetSignalToNoise()             { return GetSignalToNoise( (Int_t)0, (Float_t)INC_CUT); };


    Int_t   GetMultiplicity(Float_t);         ///< cluster multiplicity	
    Int_t   GetMultiplicity()  { return GetMultiplicity(INC_CUT);  };
    
    cTrkLevel1* GetLevel1Struct();
    
    Float_t GetCOG(Int_t);
    Float_t GetCOG(Float_t angle);
    Float_t GetCOG(){ return GetCOG(0); };
    Float_t GetETA(Int_t,float);
    Float_t GetETA(float angle){ return GetETA(0,angle); };
    
    Bool_t IsBad(Int_t);                ///< bad-cluster flag

    Int_t GetPlane() { return (Int_t)((view+1)/2);};         ///< plane number (1-6)
    Int_t GetLadder(){ return 1+(Int_t)((maxs-1)/1024);  };  ///< ladder number (1-3)
    Bool_t IsY(){ return (Bool_t)view%2; };
    Bool_t IsX(){ return !((Bool_t)view%2); };
	
    TrkCluster* GetTrkCluster(){ return this; };

    friend class TrkLevel1;

    ClassDef(TrkCluster,2);

};

/**
 * \brief Class to describe tracker LEVEL1 data.
 *
 * A Level1 tracker event is defined as a collection of clusters ( TrkCluster objects ).
 * The result of common-noise computation for each viking is also stored ( cnev[24][12] ).
 * A general flag summarize the event status (missing sections, crc failures, decoding errors ans so on...).
 */
class TrkLevel1 : public TObject {

 private:
    
 public:

    Int_t         good[12];       ///< event status
    Float_t       cn[24][12];     ///< CN
//    Float_t       cnrms[24][12];  ///< CN RMS
    Int_t         cnn[24][12];    ///< number of strips for CN computation
//	Int_t         fshower[12];
//	Int_t         good1;
//	Int_t         crc[12];
	
    TClonesArray *Cluster;        ///< clusters
    
	
    TrkLevel1();
    ~TrkLevel1(){Delete();};
    
    int nclstr() {return Cluster->GetEntries();}    ///< number of stored clusters
    
    void Dump();
    void SetFromLevel1Struct(cTrkLevel1 *);
//	void GetLevel1Struct(cTrkLevel1 *) const;
    cTrkLevel1* GetLevel1Struct();
    void Clear();
    void Delete();
    
    TrkCluster*   GetCluster(int);
    
    TrkLevel1*    GetTrkLevel1(){return this;}
    TClonesArray* GetClusters(){return Cluster;};   ///< returns pointer to the cluster array
    
    int           LoadPfaParam(TString);
    
    ClassDef(TrkLevel1,2);

};


#endif
