/**
 * \file TrkStruct.h
 * \author Elena Vannuccini
 * \date
 */
#ifndef trkstruct_h
#define trkstruct_h

#define NTRKMAX 10
#define NSINGMAX 100

#include <CalibTrk1Event.h>
#include <CalibTrk2Event.h>

#include <TString.h>
#include <TFile.h>
#include <TTree.h>

#include <fstream>
#include <iostream>
/**
 * \brief Struct to pass calibration parameters to F77 routines
 */
struct cTrkCalib {

    float pedestal[128][24][12];
    float pedestal_t[128][24][12];
    float sigma[128][24][12];
    float sigma_t[128][24][12];
    int bad[128][24][12];

    void FillACalibFrom(TFile* , Int_t , Int_t );
    void FillFCalibFrom(TFile* , Int_t , Int_t );
    void FillTCalibFrom(TFile* , Int_t , Int_t );
    void FillTCalibFrom(TString);

};
// ==================================================================
/**
 * \brief Struct to pass tracker LEVEL0 data to F77 routines
 */
struct cTrkLevel0 {
    int DAQmode[12];
    int DSPnumber[12];
    int DATAlength[12];
    int eventn[12];
    int nclust[12];
    int cutc[12];
    int cutcl[12];
    int addrcluster[3][12];
    int signcluster[3][12];
    int fc[12];
    int compressiontime[12];
    int fl5[12]; 
    int fl4[12];
    int fl3[12];
    int fl2[12];
    int fl1[12];
    int fl6[12];
    int checksum[12];
    int TOTDATAlength;
    int datatracker[49152];
    int pnum[12];    
    int cmdnum[12];
    int bid[12];
    int alarm[12];
    int aswr[12]; 
    int good0;
    int crc[12]; 
};
// ==================================================================
/**
 * \brief Struct to pass tracker LEVEL1 data to F77 routines
 */
struct cTrkLevel1 {
    int good1;
    int nclstr1;
    int view[500];
    int ladder[500];
    int maxs[500];
    int mult[500];
    float dedx[500];
    int indstart[500];
    int indmax[500];
    int totCLlength;
    float clsignal[8500];
    float cnev[24][12];  
};
// ==================================================================
/**
 * \brief Struct to pass tracker LEVEL2 data to F77 routines
 */
struct cTrkLevel2 {
    int good2;
    int crc[12];
//  ----------------------------
    int nclsx;
    int planex[NSINGMAX];
    float xs[NSINGMAX][2];
    float signlxs[NSINGMAX];
//  ----------------------------
    int nclsy;
    int planey[NSINGMAX];
    float ys[NSINGMAX][2];
    float signlys[NSINGMAX];
//  ----------------------------
    int ntrk;
    int image[NTRKMAX];
    float xm_nt[NTRKMAX][6];
    float ym_nt[NTRKMAX][6];
    float zm_nt[NTRKMAX][6];
    float resx_nt[NTRKMAX][6];
    float resy_nt[NTRKMAX][6];
    float al_nt[NTRKMAX][5];
    float coval[NTRKMAX][5][5];
    float chi2_nt[NTRKMAX];
    int xgood_nt[NTRKMAX][6];
    int ygood_nt[NTRKMAX][6];
    float xv_nt[NTRKMAX][6];
    float yv_nt[NTRKMAX][6];
    float zv_nt[NTRKMAX][6];
    float axv_nt[NTRKMAX][6];
    float ayv_nt[NTRKMAX][6];
    float dedx_x[NTRKMAX][6];
    float dedx_y[NTRKMAX][6];
    float bdl[NTRKMAX];
};
// ==================================================================
/**
 * \brief Struct to pass calibration/parameter file paths to F77 routines
 */
struct cPath {
    char path[80];
    int  pathlen;
    int  error;
/**
 * Fill the struct variables from a TString object
 */
    void FillWith(TString s){
        pathlen = s.Length();
        const char *pc = s.Data();
        for(Int_t i=0; i<=pathlen; i++) path[i] = *pc++;
    };


};

// ==================================================================
/**
 * \brief Struct to pass magnetic-field file paths to F77 routines
 */
struct cBPath {
    char b_path[80];
    int  b_pathlen;
    int  b_error;
    int  b_loaded;
/**
 * Fill the struct variables from a TString object and set 
 * the load flag to FALSE.
 */
    void FillWith(TString s){
	b_loaded  = 0;
	b_pathlen = s.Length();
	const char *pc = s.Data();
	for(Int_t i=0; i<=b_pathlen; i++) b_path[i] = *pc++;
    };
/**
 * Fill the struct variables from a TString object
 */
    int BIsLoaded(){ return b_loaded; };
	
};

// ==================================================================
/**
 * \brief Struct to set debug mode in F77 routines
 */
struct cDbg {
    int debug;
    int verbose;
    int warning;
//    bool debug;
//    bool verbose;
    void SetNone()   {debug=0; verbose=0; warning=0;};
    void SetWarning(){debug=0; verbose=0; warning=1;};
    void SetVerbose(){debug=0; verbose=1; warning=1;};
    void SetDebug()  {debug=1; verbose=1; warning=1;};
};

extern "C" {
    extern struct cTrkCalib  pedsigbad_;
    extern struct cTrkLevel0 level0event_;
    extern struct cTrkLevel1 level1event_;
    extern struct cTrkLevel2 level2event_;
    extern struct cPath      path_;
    extern struct cBPath     bpath_;
    extern struct cDbg       dbg_;
    void fillpedsigfromdefault_();
    int readmipparam_(); 
    int readchargeparam_(); 
    int readvkmask_(); 
    int readalignparam_(); 
    int readetaparam_(); 
    void reductionflight_(int*);
    int analysisflight_();
}


#endif

