/**
 * \file TrkParams.h
 * \author Elena Vannuccini
 */
#ifndef trkparams_h
#define trkparams_h

#include <TObject.h>
#include <TString.h>
#include <TSQLServer.h>
#include <TSystem.h>
#include <TROOT.h> // EMILIANO: needed to compile with ROOT > 5.16/x

#include <GLTables.h>
#include <TrkStruct.h>
#include <TrkCalib.h>

#define NTYPES 10
#define NGF 14

/**
 * \brief Class to store tracker parameter information
 * 
 * Tracker parameters include calibration parameters (PED, SIG and BAD), 
 * parameters loaded from DB (alignment, ADC-to-mip, etc. ) and some parameters 
 * to configure tracker processing (PFA, ecc...).
 *
 * Calibrations can be either those evaluated online or default values.
 * External parameter are labelled according to the DB convention:
 *
 * type  description
 * -----------------
 * 1     field
 * 2     adc-to-mip
 * 3     charge-correlation
 * 4     p.f.a. (eta)
 * 5     alignment
 * 6     VA1 mask
 * 7     default calibration
 *
 * Internal parameters are initialized to default values 
*
 * TrkParams::SetCalib(...) and TrkParams::Set(...) methods allow to set 
 * required parameters, from an input path, from the DB (for a given input 
 * run id) and from environment variable PAM_CALIB.
 * TrkParams::LoadCalib() and TrkParams::Load() methods load parameters into
 * F77 common. 
 * 
 * Tracker libraries (TrkLevel0, TrkLevel1 and TrkLevel2) implement automatic
 * setting (either from DB or from environment variables) and loading of parameters.
 * If one needs to load custom parameters, it is enough to call TrkParams::Set(...)
 * with proper arguments. All the methods are static, and can be used as standalone functions. 
 *
 * For example: 
 *
 * TrkParams::Set("/param-path/new-mip-param/",2);
 * 
 * After this instruction (to be called once), new adc-to-mip conversion parameters 
 * will be loaded automatically (once) and used until the next set instruction 
 */
class TrkParams : public TObject {

private:
	
    static Int_t GetIndex( UInt_t type );

public:

    // ---------------------------------------------
    // variables to store loaded-calibration info
    // ---------------------------------------------
    static GL_TRK_CALIB calib; 
    static TString      calibpatht; //truncated
    static TString      calibpathf; //full
    static Bool_t       calib104;
    static Bool_t       calibload;
    static TrkCalib*    calib1;
    static TrkCalib*    calib2;
    // ---------------------------------------------
    // variables to store loaded-parameters info
    // ---------------------------------------------
    static UInt_t    trkparamtype[NTYPES];
    static GL_PARAM  gl[NTYPES]; 
    static TString   glpath[NTYPES];
    static Bool_t    glload[NTYPES];
    // -----------------------------------------------
    // some parameters to configure tracker processing
    // -----------------------------------------------
    static int init__pfa;
    static int init__mini_trackmode;
    static int init__mini_istepmin;
    static double init__mini_fact;
    static cDbg init__dbg_mode; //debug mode (for both c++ and F77 roiutines)
    static float init__pfa_e234ax[6]; // angle intervals for eta2-3-4
    static float init__pfa_e234ay[6]; // angle intervals for eta2-3-4
    static double init__deltab_0; 
    static double init__deltab_1;
    static double init__dlt;

    // ------------------------------------------
    // nominal z-coordinates to define acceptance
    // ------------------------------------------
    static int   nGF;
    static float zGF[NGF];
    static float xGF_min[NGF];
    static float xGF_max[NGF];
    static float yGF_min[NGF];
    static float yGF_max[NGF];
    static TString GF_element[NGF];

    TrkParams();
       
    // ---------------------------------------------
    // calibration (PED-SIG-BAD)
    // ---------------------------------------------
    static Bool_t SetCalib( GL_RUN* , TSQLServer*  );
    static Bool_t LoadCalib( );
    static Bool_t CalibIsLoaded(){ return calibload; };
    static UInt_t ValidateTrkCalib( CalibTrk1Event* );
//    static void FillFCalibFrom(TFile* , Int_t , Int_t );//full
//    static void FillTCalibFrom(TFile* , Int_t , Int_t );//truncated
    static void FillTCalibFrom(TString path);
    static void FillACalibFrom(TFile* , Int_t , Int_t );//all
    static void FillMask(TFile* , Int_t , Int_t );

    // ---------------------------------------------
    // parameters
    // ---------------------------------------------
    static Bool_t Set( GL_RUN* glrun, TSQLServer* dbc, UInt_t type);
    static Bool_t Set( GL_RUN* glrun, TSQLServer* dbc){ 
	for(Int_t i=0; i<NTYPES; i++) 
	    if( !TrkParams::Set(glrun,dbc,trkparamtype[i]) )return false; 
	return true; 
    };
    static Bool_t Set( TString, UInt_t );
    static Bool_t Set( UInt_t );
    static Bool_t Set( );
    static Bool_t Load( UInt_t );
    static Bool_t Load( );
    static Bool_t IsLoaded( UInt_t type ){return glload[TrkParams::GetIndex(type)];};
    static Bool_t IsLoaded( );

    static GL_PARAM GetParams( UInt_t type ){ return gl[TrkParams::GetIndex(type)]; }; 

    // ---------------------------------------------
    // retrieve parameters
    // ---------------------------------------------
    static float GetBX(float*);
    static float GetBY(float*);
    static float GetBZ(float*);
    static float GetResolution(int,float);
    static int   GetPFA(){return sw_.pfaid;};
    static int GetPFA_NbinsAngle();
    static int GetPFA_NbinsETA();
    static int GetPFA_NbinsCharge(){return 0;};
    static int GetPFA_AngleBin(float);
    static float* GetPFA_ETAcorr(TString,int,int,int);
    static float* GetPFA_ETAabs(TString,int);
    static float  GetPFA_ETAlandi(int,int,int);
    static float  GetPFA_ETAlandi(int nv,int nl,float ang)
	{ return TrkParams::GetPFA_ETAlandi(nv,nl,TrkParams::GetPFA_AngleBin(ang)); };



    // ---------------------------------------------
    // debug mode 
    // ---------------------------------------------

    static void SetDebugMode()  {init__dbg_mode.SetDebug();   dbg_=init__dbg_mode;};
    static void SetVerboseMode(){init__dbg_mode.SetVerbose(); dbg_=init__dbg_mode;};
    static void SetWarningMode(){init__dbg_mode.SetWarning(); dbg_=init__dbg_mode;};    
    static void SetQuietMode()  {init__dbg_mode.SetNone();    dbg_=init__dbg_mode;};    
    static Bool_t DebugMode(){return init__dbg_mode.debug;};
    static Bool_t WarningMode(){return init__dbg_mode.warning;};
    static Bool_t VerboseMode(){return init__dbg_mode.verbose;};

    // ------------------------------------------------
    // alcuni metodi che non sapevo dove mettere 
    // per inizializzare alcune variabili in F77 common
    // ------------------------------------------------     
    // set to default values
    static void SetTrackingMode()   {track_.trackmode = init__mini_trackmode;};
    static void SetPrecisionFactor(){track_.fact      = init__mini_fact;     };
    static void SetStepMin()        {track_.istepmin  = init__mini_istepmin; };
    static void SetDeltaB()         {deltab_.delta0   = init__deltab_0;
                                     deltab_.delta1   = init__deltab_1;};    
    static void SetDLT()            {deltab_.dlt   = init__dlt;};
    static void SetMiniDefault()    {
	SetTrackingMode(); 
	SetPrecisionFactor(); 
	SetStepMin(); 
	SetDeltaB(); 
	SetDLT();
    }; //
    static void SetPFA();
    
    // set to custom values
    static void SetTrackingMode(int);
    static void SetPrecisionFactor(double);
    static void SetStepMin(int);
    static void SetDeltaB(int,double);    
    static void SetDLT(double);
    static void SetPFA(int);
    static void SetPFA(float *);
    
    // ---------------------------------------------
    // varie ed eventuali...
    // ---------------------------------------------
    static TSQLServer* SetDBConnection();


    ClassDef(TrkParams,2);

};

#endif

