/**
 * \file TrkLevel2.h
 * \author Elena Vannuccini
 */
#ifndef trklevel2_h
#define trklevel2_h

#include <TObject.h>
#include <TObjArray.h>
#include <TClonesArray.h>
#include <TRefArray.h>
#include <TRef.h>

#include <TrkParams.h>
#include <TrkLevel1.h>

// z-coordinate of track state-vector reference-plane
#define ZINI 23.5   
// (mechanical) z-coordinate of the tracker planes
#define ZTRK6 -22.22
#define ZTRK5 -13.31
#define ZTRK4 -4.41
#define ZTRK3 4.49
#define ZTRK2 13.39
#define ZTRK1 22.29
// magnet cavity dimensions
#define ZMAGNHIGH 21.83
#define ZMAGNLOW -21.83
#define XMAGNHIGH 8.07 
#define XMAGNLOW -8.07 
#define YMAGNHIGH 6.57 
#define YMAGNLOW -6.57 
// tof planes 
#define ZS11  53.74
#define ZS12  53.04
#define ZS21  23.94
#define ZS22  23.44
#define ZS31 -23.49
#define ZS32 -24.34

// (mechanical) x/y-coordinates of magnet cavity
/* #define XTRKL -8.1 */
/* #define XTRKR  8.1 */
/* #define YTRKL -6.6 */
/* #define YTRKR  6.6 */

/**
 * \brief Class to describe, by points, a particle trajectory in the apparatus. 
 *
 * The idea is to create it by integrating the equations of motion, given the 
 * track state vector and the z coordinates where to evaluate track position.
 */
// ==================================================================
class Trajectory : public TObject{
 private:

 public:

    int npoint; ///< number of evaluated points along the trajectory
    float* x;   //[npoint]
    float* y;   //[npoint]
    float* z;   //[npoint]
    float* thx; //[npoint]
    float* thy; //[npoint]
    float* tl;  //[npoint]

    Trajectory();
    Trajectory(int n);
    Trajectory(int n, float* pz);
    ~Trajectory(){Delete();};
    void Dump();
    void Delete();

    int DoTrack2(float* al);
    float GetLength(){float l=0; for(int i=0; i<npoint;i++)l=l+tl[i]; return l;};
    float GetLength(int,int);

    ClassDef(Trajectory,3);

};
/**
 * \brief Class to describe fitted tracks. 
 *
 * A track is defined by the measured coordinates associated to it, the 
 * track status vector, plus other quantities.
 * A track may have an "image", due to the ambiguity in the y view.
 *
 * Cluster flags: xgood[6], ygood[6]
 * 
 * xgood/ygood = +/- 0lsccccccc
 * ccccccc ID (1-7483647) of the included cluster  
 * s       sensor number (1,2   - increasing y)
 * l       ladder number (1,2,3 - increasing x)
 * +/-     does-not/does include bad strips
 *
 */
// ==================================================================
class TrkTrack : public TObject {

private:

public:

    int   seqno;           ///<stored track sequential number
    int   image;           ///<sequential number of track-image
	
    float al[5];           ///<TRACK STATE VECTOR 
    float coval[5][5];     ///<covariance matrix 
    int   xgood[6];        ///<cluster id for x-view (0 = view not included in the fit) 
    int   ygood[6];        ///<cluster id for y-view (0 = view not included in the fit) 
    float xm[6];           ///<measured x coordinates
    float ym[6];           ///<measured y coordinates 
    float zm[6];           ///<measured z coordinates 
    float resx[6];         ///<spatial resolution on X view
    float resy[6];         ///<spatial resolution on y view
    float tailx[6];        ///<spatial resolution tail on X view
    float taily[6];        ///<spatial resolution tail on y view
    float chi2;            ///<chi2
    int   nstep;           ///<n.step
    float xv[6];           ///<calculated x coordinates
    float yv[6];           ///<calculated y coordinates
    float zv[6];           ///<calculated z coordinates
    float axv[6];          ///<calculated angles (deg) on x view
    float ayv[6];          ///<calculated angles (deg) on y view
    float dedx_x[6];       ///<dE/dx in MIP (<0 if saturated)
    float dedx_y[6];       ///<dE/dx in MIP (<0 if saturated) 
    int   multmaxx[6];     ///<cluster multiplicity and strip of maximum on x view
    int   multmaxy[6];     ///<cluster multiplicity and strip of maximum on y view
    float seedx[6];        ///< seed of the cluster x
    float seedy[6];        ///< seed of the cluster y
    float xpu[6];          ///< x coordinate in pitch units
    float ypu[6];          ///< y coordinate in pitch units

    float xGF[14];         ///<calculated x coordinates on GF reference planes
    float yGF[14];         ///<calculated y coordinates on GF reference planes

    TrkTrack();
    TrkTrack(const TrkTrack&);

    ~TrkTrack(){ Delete(); };
	
    void Dump();
    void Clear();
    void Clear(Option_t *option){Clear();};
    void Delete();
    void Copy(TrkTrack&);
//    void Set();

    Int_t  GetSeqNo(){return seqno;}        ///< Returns the track sequential number
    Int_t  GetImageSeqNo(){return image;}   ///< Returns the track image sequential number
    Bool_t HasImage(){return !(image==-1);} ///< Returns true if the track has an image
    int DoTrack(Trajectory* t);   	    ///< Evaluates the trajectory in the apparatus.
    int DoTrack2(Trajectory* t);   	    ///< Evaluates the trajectory in the apparatus.
    float BdL(){return 0;};		    ///< Evaluates the integral of B*dL along the track.
    Int_t GetNX(){Int_t n=0; for(Int_t i=0; i<6; i++)n+=(Int_t)XGood(i); return n;}; 
    Int_t GetNY(){Int_t n=0; for(Int_t i=0; i<6; i++)n+=(Int_t)YGood(i); return n;};
    Int_t GetNtot(){return GetNX()+GetNY();};
    Float_t GetRigidity();
    Float_t GetDeflection();
    Bool_t IsSaturated(int,int);
    Bool_t IsSaturated(int);
    Bool_t IsSaturated();
    Bool_t IsBad(int,int);
    Float_t GetDEDX();
    Float_t GetDEDX(int ip);
    Float_t GetDEDX(int ip,int iv);
    Int_t GetLeverArmX();
    Int_t GetLeverArmY();
    Float_t GetChi2X();
    Float_t GetChi2Y();
    Float_t GetLnLX();
    Float_t GetLnLY();

    Float_t GetEffectiveAngle(int ip, int iv);
    
    void SetMeasure(double *xmeas, double *ymeas, double *zmeas);
    void SetResolution(double *rx, double *ry);
    void SetTail(double *tx, double *ty, double factor);
    void SetStudentParam(int flag);
    void SetGood(int *xg, int *yg);
    void LoadField(TString s);
    void Fit(double pfixed, int& fail, int iprint, int froml1);
    void Fit(double pfixed, int& fail, int iprint){ Fit(pfixed,fail,iprint,0); };
    void FitReset();
    void SetTrackingMode(int trackmode);
    void SetPrecisionFactor(double fact);
    void SetStepMin(int istepmin);
    void SetDeltaB(int id, double db);

    Bool_t IsInsideCavity(float);
    Bool_t IsInsideCavity(){ return IsInsideCavity(0.); };
    Bool_t IsInsideAcceptance();

    Bool_t EvaluateClusterPositions();

    void FillMiniStruct(cMini2track&);
    void SetFromMiniStruct(cMini2track*);
    
    Int_t GetClusterX_ID(int ip);
    Int_t GetClusterY_ID(int ip);
    Int_t GetLadder(int ip);
    Int_t GetSensor(int ip);
    Bool_t XGood(int ip){ return GetClusterX_ID(ip)!=-1; };
    Bool_t YGood(int ip){ return GetClusterY_ID(ip)!=-1; };
    void ResetXGood(int ip){ xgood[ip]=0; };
    void ResetYGood(int ip){ ygood[ip]=0; };
    void SetXGood(int ip, int clid, int is);
    void SetYGood(int ip, int clid, int is);

    Bool_t BadClusterX(int ip){ return IsBad(ip,0); };
    Bool_t BadClusterY(int ip){ return IsBad(ip,1); };

    Bool_t SaturatedClusterX(int ip){ return IsSaturated(ip,0); };
    Bool_t SaturatedClusterY(int ip){ return IsSaturated(ip,1); };

    Int_t GetClusterX_Multiplicity(int ip){ return (Int_t)(multmaxx[ip]/10000); };
    Int_t GetClusterY_Multiplicity(int ip){ return (Int_t)(multmaxy[ip]/10000); };
    Int_t GetClusterX_MaxStrip(int ip){ return (Int_t)(multmaxx[ip]%10000); };
    Int_t GetClusterY_MaxStrip(int ip){ return (Int_t)(multmaxy[ip]%10000); };
    Float_t GetClusterX_Seed(int ip){ return seedx[ip]; };
    Float_t GetClusterY_Seed(int ip){ return seedy[ip]; };
/*     Float_t GetClusterX_CoordinatePU(int ip); */
/*     Float_t GetClusterY_CoordinatePU(int ip); */
    
    Float_t GetYav();
    Float_t GetXav();
    Float_t GetZav();

    Int_t GetNColumns();

    Float_t GetDEDX_max(int ip, int iv);
    Float_t GetDEDX_max(int iv){ return GetDEDX_max(-1,iv); };
    Float_t GetDEDX_max(){ return GetDEDX_max(-1,-1); };
    Float_t GetDEDX_min(int ip, int iv);
    Float_t GetDEDX_min(int iv){ return GetDEDX_min(-1,iv); };
    Float_t GetDEDX_min(){ return GetDEDX_min(-1,-1); };

    Float_t GetResidual_max(int ip, int iv);
    Float_t GetResidual_max(int iv){ return GetResidual_max(-1,iv); };
    Float_t GetResidual_max(){ return GetResidual_max(-1,-1); };

    Int_t GetClusterX_Multiplicity_max();
    Int_t GetClusterX_Multiplicity_min();
    Int_t GetClusterY_Multiplicity_max();
    Int_t GetClusterY_Multiplicity_min();

    Float_t GetClusterX_Seed_min();
    Float_t GetClusterY_Seed_min();

    TrkTrack* GetTrkTrack(){return this;};

    friend class TrkLevel2;

    ClassDef(TrkTrack,5);

};
/**
 * \brief Class to describe single clusters ("singlets"). 
 *
 * Single clusters are clusters not associated to any track.
 */
class TrkSinglet : public TObject {

private:
	

public:
	
    int plane;       ///<plane 
    float coord[2];  ///<coordinate (on sensor 1 and 2)
    float sgnl;      ///<cluster signal in MIP (<0 if saturated)
    int multmax;     ///<cluster multiplicity and strip of maximum

    TrkSinglet();
    TrkSinglet(const TrkSinglet&);
    ~TrkSinglet(){Delete();};

    void Dump();
    void Clear();
    void Clear(Option_t *option){Clear();};
    void Delete(){Clear();};
    Float_t GetSignal(){return fabs(sgnl);}
    Bool_t IsSaturated(){return (sgnl<0); };

    Bool_t IsBad()                 { return multmax<=0; };
    Int_t GetCluster_Multiplicity(){ return (Int_t)(abs(multmax)/10000); };
    Int_t GetCluster_MaxStrip()    { return (Int_t)(abs(multmax)%10000); };


    friend class TrkLevel2;

    ClassDef(TrkSinglet,4);

};

/**
 * \brief Class to describe tracker LEVEL2 data.
 *
 * A tracker events is defined by some general variables, plus the collection of all the fitted tracks and all 
 * single clusters on X and Y views. 
 * Tracks and single clusters ("singlets") are described by the classes TrkTrack and TrkSinglet respectivelly.
 * 
 * Each track may have an "image", due to the ambiguity on the Y view, which is stored also. 
 * Thus, the number of stored tracks ( ntrk() ) differs from the number of "physical" tracks ( GetNTracks() ). 
 * Proper methods allow to sort tracks and select the physical ones ( GetTracks() ).
 *
 * The event status indicates the processing status of data from each DSP, according to the following
 * notation:
 *
 * LSB --> 0 missing packet
 *         1 CRC error
 *         2 on-line software alarm (latch-up, timeout ecc...)
 *         3 jump in the trigger counter
 *         4 decode error
 *         5 n.clusters > maximum number (level1 processing)
 *         6 
 *         7 
 *         8 n.clusters > maximum value (level2 processing)
 *         9 n.couples per plane > maximum values (vector dimention)
 *         10 n.doublets > maximum values
 *         11 n.triplets > maximum values
 *         12 n.yz-clouds > maximum values
 *         13 n.xz-clouds > maximum values 
 *         14 n.candidate-tracks > maximum values
 *         15 n.couples per plane > maximum values (for Hough transform)
 * MSB --> 16 
 *         
 *
 * For all data processed before June 2007 the event status was coded according to
 * a different rule:
 *
 * Status of level1 processing
 *  0 -- OK  
 *  1 -- missing packet
 *  2 -- 1  CRC error
 *  3 -- 2 on-line software alarm (latch-up flags asserted or n.transmitted-words = 0)
 *  4 -- 3 jump in the trigger counter
 * 10 -- 4 decode error
 * 11 -- 5  n.clusters > maximum number (for level1 processing)
 * Status of level2 processing
 * 21 -- 0 n.clusters > maximum value (for level2 processing)
 * 22 -- 1 n.couples per plane > maximum values (vector dimention)
 * 23 -- 2 n.doublets > maximum values 
 * 24 -- 3 n.triplets > maximum values 
 * 25 -- 4 n.yz-clouds > maximum values 
 * 26 -- 5 n.xz-clouds > maximum values 
 * 27 -- 6 n.candidate-tracks > maximum values 
 * 28 -- 7 n.couples per plane > maximum values (for Hough transform)
 *  
 * 
 */
class TrkLevel2 : public TObject {

 private:
 
 public:

    Int_t         good[12];       ///< event status
    UInt_t        VKmask[12];     ///< Viking-chip mask
    UInt_t        VKflag[12];     ///< Viking-chip flag

    TClonesArray *Track;        ///< fitted tracks
    TClonesArray *SingletX;     ///< x singlets
    TClonesArray *SingletY;     ///< y singlets

    TrkLevel2();
//    TrkLevel2(cTrkLevel2 *);
    ~TrkLevel2(){Delete();};
	
    void Clear();
    void Clear(Option_t *option){Clear();};
    void Delete();
    void Set();
    
    int ntrk() {return Track->GetEntries();}    ///< number of stored track
    int nclsx(){return SingletX->GetEntries();} ///< number of x singlets 
    int nclsy(){return SingletY->GetEntries();} ///< number of y singlets 

    void Dump();
    void SetFromLevel2Struct(cTrkLevel2 *, TrkLevel1 *);
    void SetFromLevel2Struct(cTrkLevel2 *s2){ SetFromLevel2Struct(s2, NULL);          };
    void SetFromLevel2Struct(TrkLevel1 *l1) { SetFromLevel2Struct(&level2event_, l1); };    
    void SetFromLevel2Struct()              { SetFromLevel2Struct(&level2event_);     };    
    void GetLevel2Struct(cTrkLevel2 *) const;
    void LoadField(TString);
    float GetBX(float* v){return TrkParams::GetBX(v);};///< Bx (kGauss)
    float GetBY(float* v){return TrkParams::GetBY(v);};///< By (kGauss)
    float GetBZ(float* v){return TrkParams::GetBZ(v);};///< Bz (kGauss)
    Float_t GetZTrk(Int_t);
    Float_t GetXTrkLeft(){return XMAGNLOW;};
    Float_t GetXTrkRight(){return XMAGNHIGH;};
    Float_t GetYTrkLeft(){return YMAGNLOW;};
    Float_t GetYTrkRight(){return YMAGNHIGH;};
    
    Bool_t IsMaskedVK(int,int);
    Bool_t GetVKMask(int,int);
    Bool_t GetVKFlag(int,int);

    TrkSinglet   *GetSingletX(int);
    TrkSinglet   *GetSingletY(int);
    
    TrkTrack     *GetStoredTrack(int i);
    Int_t         GetSeqNo(Int_t i)  {return (((TrkTrack *)Track->At(i))->seqno);}; ///< Returns track sequential number

    TRefArray *GetTracks_NFitSorted();
    TRefArray *GetTracks(){return this->GetTracks_NFitSorted();};
    
    Int_t     GetNTracks();
    TrkTrack* GetTrack(int i);
    TrkTrack* GetTrackImage(int i);
    
    TrkLevel2*    GetTrkLevel2(){return this;}
    TClonesArray* GetTrackArray(){return Track;};///< returns pointer to the track array
    
    void   StatusDump(int view);
    Bool_t StatusCheck(int view, int flagmask);

    ClassDef(TrkLevel2,3);

};

#endif
