/**
 * \file TrkParams.h
 * \author Elena Vannuccini
 */
#ifndef trkparams_h
#define trkparams_h

#include <TObject.h>
#include <TString.h>
#include <TSQLServer.h>
#include <TSystem.h>

#include <GLTables.h>
#include <TrkStruct.h>
#include <TrkCalib.h>

#define NTYPES 10

/**
 * \brief Class to store tracker parameter information
 * 
 * Tracker parameters include calibration parameters (PED, SIG and BAD) and 
 * other parameters (alignment, ADC-to-mip, etc. ).
 * The first can be either those evaluated online or default values.
 * The second are labelled according to the DB convention:
 *
 * type  description
 * -----------------
 * 1     field
 * 2     adc-to-mip
 * 3     charge-correlation
 * 4     p.f.a. (eta)
 * 5     alignment
 * 6     VA1 mask
 * 7     default calibration
 *
 * TrkParams::SetCalib(...) and TrkParams::Set(...) methods allow to set 
 * required parameters, from an input path, from the DB (for a given input 
 * run id) and from environment variable PAM_CALIB.
 * TrkParams::LoadCalib() and TrkParams::Load() methods load parameters into
 * F77 common. 
 * 
 * Tracker libraries (TrkLevel0, TrkLevel1 and TrkLevel2) implement automatic
 * setting (either from DB or from environment variables) and loading of parameters.
 * If one needs to load custom parameters, it is enough to call TrkParams::Set(...)
 * with proper arguments. All the methods are static, and can be used as standalone functions. 
 *
 * For example: 
 *
 * TrkParams::Set("/param-path/new-mip-param/",2);
 * 
 * After this instruction (to be called once), new adc-to-mip conversion parameters 
 * will be loaded automatically (once) and used until the next set instruction 
 */
class TrkParams : public TObject {

private:
	
    static Int_t GetIndex( UInt_t type );

public:

    // ---------------------------------------------
    // variables to store loaded-calibration info
    // ---------------------------------------------
    static GL_TRK_CALIB calib; 
    static TString      calibpatht; //truncated
    static TString      calibpathf; //full
    static Bool_t       calib104;
    static Bool_t       calibload;
    // ---------------------------------------------
    // variables to store loaded-parameters info
    // ---------------------------------------------
    static UInt_t    trkparamtype[NTYPES];
    static GL_PARAM  gl[NTYPES]; 
    static TString   glpath[NTYPES];
    static Bool_t    glload[NTYPES];
    
    TrkParams();
    
    // ---------------------------------------------
    // calibration (PED-SIG-BAD)
    // ---------------------------------------------
    static Bool_t SetCalib( GL_RUN* , TSQLServer*  );
    static Bool_t LoadCalib( );
    static Bool_t CalibIsLoaded(){ return calibload; };
    static UInt_t ValidateTrkCalib( CalibTrk1Event* );
    static void FillFCalibFrom(TFile* , Int_t , Int_t );//full
    static void FillTCalibFrom(TFile* , Int_t , Int_t );//truncated
    static void FillTCalibFrom(TString path);
    static void FillACalibFrom(TFile* , Int_t , Int_t );//all
    static void FillMask(TFile* , Int_t , Int_t );

    // ---------------------------------------------
    // parameters
    // ---------------------------------------------
    static Bool_t Set( GL_RUN* glrun, TSQLServer* dbc, UInt_t type);
    static Bool_t Set( GL_RUN* glrun, TSQLServer* dbc){ 
	for(Int_t i=0; i<NTYPES; i++) 
	    if( !TrkParams::Set(glrun,dbc,trkparamtype[i]) )return false; 
	return true; 
    };
    static Bool_t Set( TString, UInt_t );
    static Bool_t Set( UInt_t );
    static Bool_t Set( );
    static Bool_t Load( UInt_t );
    static Bool_t Load( );
    static Bool_t IsLoaded( UInt_t type ){return glload[TrkParams::GetIndex(type)];};
    static Bool_t IsLoaded( );


    // ---------------------------------------------
    // debug mode 
    // ---------------------------------------------
    static cDbg dbg_mode; //debug mode (for both c++ and F77 roiutines)

    static void SetDebugMode()  {dbg_mode.SetDebug();   dbg_=dbg_mode;};
    static void SetVerboseMode(){dbg_mode.SetVerbose(); dbg_=dbg_mode;};
    static void SetWarningMode(){dbg_mode.SetWarning(); dbg_=dbg_mode;};    
    static Bool_t DebugMode(){return dbg_mode.debug;};
    static Bool_t WarningMode(){return dbg_mode.warning;};
    static Bool_t VerboseMode(){return dbg_mode.verbose;};

    ClassDef(TrkParams,2);

};

#endif

