/**
 * \file TrkLevel2.h
 * \author Elena Vannuccini
 */
#ifndef trklevel2_h
#define trklevel2_h

#include <TObject.h>
#include <TObjArray.h>
#include <TClonesArray.h>
#include <TRefArray.h>
#include <TRef.h>

#include <TrkStruct.h>
#include <TrkLevel1.h>

// z-coordinate of track state-vector reference-plane
#define ZINI 23.5   
// upper and lower (mechanical) z-coordinate of the tracker
//#define ZTRKUP 22.29
//#define ZTRKDW -22.22
// (mechanical) z-coordinate of the tracker planes
#define ZTRK6 -22.23
#define ZTRK5 -13.32
#define ZTRK4 -4.42
#define ZTRK3 4.48
#define ZTRK2 13.38
#define ZTRK1 22.28
// (mechanical) x/y-coordinates of magnet cavity
#define XTRKL -8.1
#define XTRKR  8.1
#define YTRKL -6.6
#define YTRKR  6.6

/**
 * \brief Class to describe, by points, a particle trajectory in the apparatus. 
 *
 * The idea is to create it by integrating the equations of motion, given the 
 * track state vector and the z coordinates where to evaluate track position.
 */
// ==================================================================
class Trajectory : public TObject{
 private:

 public:

    int npoint; ///< number of evaluated points along the trajectory
    float* x;   ///< x coordinates 
    float* y;   ///< y coordinates 
    float* z;   ///< z coordinates 
    float* thx; ///< x projected angle 
    float* thy; ///< y projected angle
    float* tl;  ///< track length

    Trajectory();
    Trajectory(int n);
    Trajectory(int n, float* pz);
    void Dump();

    float GetLength(){float l=0; for(int i=0; i<npoint;i++)l=l+tl[i]; return l;};
    float GetLength(int,int);

    ClassDef(Trajectory,1);

};
/**
 * \brief Class to describe fitted tracks. 
 *
 * A track is defined by the measured coordinates associated to it, the 
 * track status vector, plus other quantities.
 * A track may have an "image", due to the ambiguity in the y view.
 */
// ==================================================================
class TrkTrack : public TObject {

private:

    int   seqno;           ///<stored track sequential number
    int   image;           ///<sequential number of track-image

	
public:

//	TRef clx[6];
//	TRef cly[6];
	TRefArray *clx;
	TRefArray *cly;

    float al[5];           ///<TRACK STATE VECTOR 
    float coval[5][5];     ///<covariance matrix 
    int   xgood[6];        ///<mask of included x planes 
    int   ygood[6];        ///<mask of included y planes 
    float xm[6];           ///<measured x coordinates
    float ym[6];           ///<measured y coordinates 
    float zm[6];           ///<measured z coordinates 
    float resx[6];         ///<spatial resolution on X view
    float resy[6];         ///<spatial resolution on y view
    float chi2;            ///<chi2
    float xv[6];           ///<calculated x coordinates
    float yv[6];           ///<calculated y coordinates
    float zv[6];           ///<calculated z coordinates
    float axv[6];          ///<calculated angles (deg) on x view
    float ayv[6];          ///<calculated angles (deg) on y view
    float dedx_x[6];       ///<signal in MIP (scaled to 300 micrometer)
    float dedx_y[6];       ///<signal in MIP (scaled to 300 micrometer)


    TrkTrack();
    TrkTrack(const TrkTrack&);

    void Dump();

    Int_t  GetSeqNo(){return seqno;}        ///< Returns the track sequential number
    Int_t  GetImageSeqNo(){return image;}   ///< Returns the track image sequential number
    Bool_t HasImage(){return !(image==-1);} ///< Returns true if the track has an image
    int DoTrack(Trajectory* t);   			///< Evaluates the trajectory in the apparatus.
    int DoTrack2(Trajectory* t);   			///< Evaluates the trajectory in the apparatus.
    float BdL(){return 0;};					///< Evaluates the integral of B*dL along the track.
    Int_t GetNX(){Int_t n=0; for(Int_t i=0; i<6; i++)n+=xgood[i]; return n;}; 
    Int_t GetNY(){Int_t n=0; for(Int_t i=0; i<6; i++)n+=ygood[i]; return n;};
    Int_t GetNtot(){return GetNX()+GetNY();};
    Float_t GetRigidity();
    Float_t GetDeflection();
    Float_t GetDEDX();

	TrkCluster *GetClusterX(int ip){TrkCluster *pt = (TrkCluster*)(clx->At(ip)); return pt;};
	TrkCluster *GetClusterY(int ip){TrkCluster *pt = (TrkCluster*)(cly->At(ip)); return pt;};
	
    TrkTrack* GetTrkTrack(){return this;};

    friend class TrkLevel2;

    ClassDef(TrkTrack,1);

};
/**
 * \brief Class to describe single clusters ("singlets"). 
 *
 * Single clusters are clusters not associated to any track.
 */
class TrkSinglet : public TObject {

private:
	

public:
	
	TRef cls;

    int plane;       ///<plane 
    float coord[2];  ///<coordinate (on sensor 1 and 2)
    float sgnl;      ///<cluster signal in MIP 

    TrkSinglet();
    TrkSinglet(const TrkSinglet&);

    void Dump();
	
	TrkCluster *GetCluster(){TrkCluster *pt = (TrkCluster*)cls.GetObject(); return pt;};

    friend class TrkLevel2;

    ClassDef(TrkSinglet,1);

};

/**
 * \brief Class to describe tracker LEVEL2 data.
 *
 * A tracker events is defined by some general variables, plus the collection of all the fitted tracks and all 
 * single clusters on X and Y views. 
 * Tracks and single clusters ("singlets") are described by the classes TrkTrack and TrkSinglet respectivelly.
 * 
 * Each track may have an "image", due to the ambiguity on the Y view, which is stored also. 
 * Thus, the number of stored tracks ( ntrk() ) differs from the number of "physical" tracks ( GetNTracks() ). 
 * Proper methods allow to sort tracks and select the physical ones ( GetTracks() ).
 */
class TrkLevel2 : public TObject {

 private:
	
//	TRefArray    *PhysicalTrack;  ///< physical tracks (no image) - 
	 
 public:

    Int_t good2;
    Int_t crc[12];

    TClonesArray *Track;        ///< fitted tracks
    TClonesArray *SingletX;     ///< x singlets
    TClonesArray *SingletY;     ///< y singlets

    TrkLevel2();
//    TrkLevel2(cTrkLevel2 *);

    int ntrk() {return Track->GetEntries();}    ///< number of stored track
    int nclsx(){return SingletX->GetEntries();} ///< number of x singlets 
    int nclsy(){return SingletY->GetEntries();} ///< number of y singlets 

    void Dump();
    void SetFromLevel2Struct(cTrkLevel2 *);
	void SetFromLevel2Struct(cTrkLevel2 *, TrkLevel1 *);
	void GetLevel2Struct(cTrkLevel2 *) const;
    void Clear();
    void LoadField(TString);
    Float_t GetZTrk(Int_t);
    Float_t GetXTrkLeft(){return XTRKL;};
    Float_t GetXTrkRight(){return XTRKR;};
    Float_t GetYTrkLeft(){return YTRKL;};
    Float_t GetYTrkRight(){return YTRKR;};
    
    TrkSinglet   *GetSingletX(int);
    TrkSinglet   *GetSingletY(int);
    
    TrkTrack     *GetStoredTrack(int i);
    Int_t         GetSeqNo(Int_t i)  {return (((TrkTrack *)Track->At(i))->seqno);}; ///< Returns track sequential number
//    TClonesArray *GetTracks_Chi2Sorted();
//    TClonesArray *GetTracks_NFitSorted();
//    TClonesArray *GetTracks();
	TRefArray *GetTracks_NFitSorted();
	TRefArray *GetTracks(){return this->GetTracks_NFitSorted();};

//    int       GetNTracks(){return this->GetTracks()->GetEntries();}
	Int_t     GetNTracks();
	TrkTrack* GetTrack(int i);
    TrkTrack* GetTrackImage(int i);

    TrkLevel2*    GetTrkLevel2(){return this;}
    TClonesArray* GetTrackArray(){return Track;};///< returns pointer to the track array
    
    ClassDef(TrkLevel2,1);

};

#endif
