/**
 * \file ToFLevel2.h
 * \author Gianfranca DeRosa / Wolfgang Menn
 */

#ifndef ToFLevel2_h
#define ToFLevel2_h
//
#include <TObject.h>
#include <TArrayI.h>
#include <TArrayF.h>
#include <TClonesArray.h>

#include <math.h> // EMILIANO

#include <ToFStruct.h>


//
// class which contains track related variables
//
#define ZTOF11 53.74
#define ZTOF12 53.04
#define ZTOF21 23.94
#define ZTOF22 23.44
#define ZTOF31 -23.49
#define ZTOF32 -24.34


/**
 * \brief  Class which contains the PMT data
 *
 * If there is a valid ADC or a TDC value (value<4095) for a PMT, both ADC and TDC data
 * are stored in the PMT class.
 * Look in the ToFLevel2Ex.cxx example in the repository how to read the PMT class. 
 */
class ToFPMT : public TObject {

 private:

 public:
    Int_t pmt_id;     ///<the identification number of the PMT from 0 to 47
    Float_t adc;      ///<raw ADC value for this PMT 
    Float_t tdc;      ///<raw TDC value for this PMT
    Float_t tdc_tw;   ///<time-walk corrected TDC value for this PMT
    //
    ToFPMT();
    ToFPMT(const ToFPMT&);
    //
    ToFPMT* GetToFPMT(){return this;};
    void Clear(Option_t *t="");



    ClassDef(ToFPMT,2);
};


/**
 * \brief Class which contains the tracker related variables
 *
 * We can use the ToF standalone to find hitted paddles, calculate beta, etc..
 * These results are then stored with the "trkseqno" = -1.
 * If we use the track from the tracker, then the penetration points in the
 * scintillators are calculated, which defines the hitted paddles. For these paddles
 * we calculate then all the output.
 * Note: The artificial ADC values are stored as dEdx in the output, the dEdx will be
 * by definition = 1. However, the artificial TDC values are just used internally
 * and not stored in the output. But one can see in both cases which PMT has artificial
 * values using "adcflag" and "tdcflag".
 * Look in the ToFLevel2Ex.cxx example in the repository how to read the tracker related
 * variables. 
 */
class ToFTrkVar : public TObject {

 private:

 public:
  //
  Int_t trkseqno; ///< tracker sequ. number: -1=ToF standalone, 0=first Tracker track, ...
  //
  Int_t npmttdc;  ///<number of the TDC measurements used to evaluate beta
  TArrayI pmttdc;  ///<contains the ID (0..47) for the PMT used to evaluate beta
  TArrayI tdcflag; ///<flag for artificial TDC, "0" if normal TDC value

/**
 * \brief beta, 12 measurements for the 12  combinations, beta[13] is modified weighted mean
 *
 * The 12 measurements are S11-S31, S11-S32, S12-S31, S12-S32, and then analogue for
 * S2-S3 and S1-S2. 
 * The calculation of beta[13] is now modified:
 * We check the individual weights for artificial TDC values, then calculate
 * am mean beta for the first time. In a second step we loop again through
 * the single measurements, checking for the residual from the mean
 * The cut on the residual reject measurements > "x"-sigma. A chi2 value is
 * calculated, furthermore a "quality" value by adding the weights which
 * are finally used. If all measurements are taken, "quality" will be = 505.
 * A chi2 cut around 3-4 and a quality-cut > 400 is needed for clean beta
 * The Level2 beta[12] which is derived in the fortran routines uses: 10.,200.,20.
 * This is not a very high quality measurement. One can re-calculate a new beta[13]
 * using the L2-method "CalcBeta"
 */
  Float_t beta[13]; 
  //
  Int_t npmtadc;  ///<number of the ADC measurements used for dEdx evaluation
  TArrayI pmtadc; ///<contains the ID (0..47) for the PMT used to evaluate dEdx
  TArrayI adcflag; ///<flag for artificial ADCs, "0" if normal ADC value
  TArrayF dedx;    ///<energy loss for this PMT in mip
  //
  Float_t xtofpos[3];  ///<x-measurement using the TDC values and the calibration from S12, S21, S32
  Float_t ytofpos[3];  ///<x-measurement using the TDC values and the calibration from S11, S22, S31
  //
  Float_t xtr_tof[6];  ///<x-measurement in the ToF layers from tracker
  Float_t ytr_tof[6];  ///<x-measurement in the ToF layers from tracker
  //
  ToFTrkVar();
  ToFTrkVar(const ToFTrkVar&);

  ToFTrkVar* GetToFTrkVar(){return this;};
  void Clear(Option_t *t="");

  ClassDef(ToFTrkVar,1);
  //
};

/**
 * \brief  Class to describe ToF LEVEL2 data
 *
 */

class ToFLevel2 : public TObject {
 private:

 public:
  //
  TClonesArray *PMT; ///<class needed to store PMT hit informations
  TClonesArray *ToFTrk; ///<track related variable class
  Int_t tof_j_flag[6];  ///<number of hitted paddle(s) for each ToF layer: flag = flag + 2**(paddlenumber-1)
  //
  Int_t unpackError;///< zero if no error presente
  Int_t default_calib; ///< one if the default calibration has been used to process the data, zero otherwise
  //
  Float_t GetdEdx(Int_t notrack, Int_t plane, Int_t adcfl); // gf Apr 07

  Float_t CalcBeta(Int_t notrack, Float_t resmax, Float_t qualitycut, Float_t chi2cut);  //  wm feb 08

  //
//  Float_t CalcBeta(Int_t notrack, Float_t resmax, Float_t chi2cut, Float_t qualitycut);   // wm feb 08
  //
  // methods to make life simplier during the analysis, returns a pointer to the ToFTrkVar class containing track related variables
  //
  Int_t ntrk(){return ToFTrk->GetEntries();};
  Int_t npmt(){return PMT->GetEntries();};

  //
  void GetLevel2Struct(cToFLevel2 *) const;
  //
  ToFTrkVar *GetToFTrkVar(Int_t notrack);
  ToFPMT *GetToFPMT(Int_t nohit);
  Int_t GetPMTid(Int_t gg, Int_t hh);
  TString GetPMTName(Int_t ind);
  
  Int_t GetPlaneIndex(Int_t pmt_id);
  void GetMatrix(Int_t notrack, Float_t adc[4][12], Float_t tdc[4][12]);
  void GetPMTIndex(Int_t pmt_id, Int_t &gg, Int_t &hh);

  // gf Apr 07
  void GetdEdxPaddle(Int_t notrack, Int_t paddleid, Int_t adcfl, Float_t &PadEdx, Int_t &SatWarning); // gf Apr 07
  TString GetPMTName(Int_t ind, Int_t &iplane, Int_t &ipaddle,Int_t &ipmt);
  Int_t GetPaddleIdOfTrack(Float_t xtr, Float_t ytr, Int_t plane); // gf Apr 07
  void GetPMTPaddle(Int_t pmt_id, Int_t &plane, Int_t &paddle); // gf Apr 07
  void GetPaddlePMT(Int_t paddle, Int_t &pmtleft, Int_t &pmtright); // gf Apr 07
  void GetPaddleGeometry(Int_t plane, Int_t paddle, Float_t &xleft, Float_t &xright, Float_t &yleft, Float_t &yright); // gf Apr 07
  Int_t GetPaddleid(Int_t plane, Int_t paddle);
  void GetPaddlePlane(Int_t padid, Int_t &plane, Int_t &paddle);
  Int_t GetNPaddle(Int_t plane);
  //
  //
  
  //
  // constructor
  //
  ToFLevel2();
  ~ToFLevel2(){Delete();}; //ELENA
  void Delete(Option_t *t=""); //ELENA
  void Set();//ELENA
  //
  //
  ToFLevel2*   GetToFLevel2(){return this;};

/**
 * Method to get the z-position of the 6 TOF layers from the plane ID
 * @param plane_id Plane ID (11 12 21 22 31 32)
 */
  Float_t      GetZTOF(Int_t plane_id){
      switch(plane_id){
      case 11: return ZTOF11;
      case 12: return ZTOF12;
      case 21: return ZTOF21;
      case 22: return ZTOF22;
      case 31: return ZTOF31;
      case 32: return ZTOF32;
      default: return 0.;
      };
  };

    //
    // Paddles position
    //
    /*
      S11 8 paddles  33.0 x 5.1 cm
      S12 6 paddles  40.8 x 5.5 cm
      S21 2 paddles  18.0 x 7.5 cm
      S22 2 paddles  15.0 x 9.0 cm
      S31 3 paddles  15.0 x 6.0 cm
      S32 3 paddles  18.0 x 5.0 cm
    */

    Int_t  GetToFPlaneID(Int_t ip);
    Int_t  GetToFPlaneIndex(Int_t plane_id);
    Bool_t HitPaddle(Int_t ,Int_t);
    Int_t  GetNHitPaddles(Int_t plane);
    void Clear(Option_t *t="");
    //
    ClassDef(ToFLevel2,3);
};

#endif

