/**
 * \file inc/CaloLevel2.h
 * \author Emiliano Mocchiutti
 */
#ifndef CaloLevel2_h
#define CaloLevel2_h
//
#include <TObject.h>
#include <TClonesArray.h>
#include <TArrayI.h>
//
#include <math.h>
//
#include <CaloStruct.h>
//

/**
 * \brief Calorimeter track-related variables class
 *
 * This class contains track-related variables. One set of variables is saved for any given 
 * track, including seltrigger event tracks. 
 *
 */
class CaloTrkVar : public TObject {

private:

public:
    //
    Int_t trkseqno; ///< this variable determine which track and which routine was used to obtain track related variables: if >= 0 standard routine/tracker track, -1 selftrigger event routine/calorimeter track, -2 high Z nuclei routine/calorimeter track, -3 standard routine/calorimeter track
    //
    // track related variables
    //
    Int_t ncore;     ///< SUM(j=1,2)SUM(i=1,PLmax) Nhit(i,j)*i , where Nhit(i,j) is the number of hits in a cylinder of radius 2 Rm (Moliere radius) around the track in the i-th plane (where the top plane is number 1 and the sum runs up to plane number PLmax, closest to the calculated electromagnetic shower maximum of the j-th view)
    Int_t noint;      ///< SUM(j=1,2)SUM(i=1,22) TH(i,j)*i , where TH(i,j) = 1 if the i-th plane of the j-th view has a cluster along (less than 4 mm away) the track with a deposited energy typical of a proton (order of one MIP), otherwise TH(i,j) = 0
    Int_t ncyl;      ///< the number of strip hit in a cylinder of radius 8 strips around the shower axis
    Int_t nlast;     ///< the same as "ncyl" but only for the last four planes and radius 4 strips.
    Int_t npre;      ///< the same as "ncyl" but only for the first three planes
    Int_t npresh;    ///< the same as "ncyl" but with radius 2 strips and only in the first four planes
    Int_t ntr;       ///< the same as "ncyl" but with radius 4 strips
    Int_t planetot;  ///< number of planes used to calculate the energy truncated mean "qmean"
    Int_t nlow;      ///< the same as "nstrip" but only after the calculated electromagnetic shower maximum
    Int_t tibar[22][2];  ///< strip traversed by the trajectory as measured by the tracker or by the selftrigger when trkseqno = -1
    Float_t tbar[22][2]; ///< position in cm as measured by the tracker or by the selftrigger when trkseqno = -1
    Float_t qcore;   ///< SUM(j=1,2)SUM(i=1,PLmax) Qhit(i,j)*i , where Qhit(i,j) is the energy released (MIP) in a cylinder of radius 2 Rm (Moliere radius) around the track in the i-th plane (where the top plane is number 1 and the sum runs up to plane number PLmax, closest to the calculated electromagnetic shower maximum of the j-th view).
    Float_t qcyl;    ///< the measured energy deposited in a cylinder of radius 8 strips around the shower axis
    Float_t qlast;   ///< the same as "qcyl" but only for the last four planes and radius 4 strips.
    Float_t qpre;    ///< the same as "qcyl" but only for the first three planes
    Float_t qpresh;  ///< the same as "qcyl" but with radius 2 strips and only in the first four planes
    Float_t qtr;     ///< the same as "qcyl" but with radius 4 strips
    Float_t qtrack;  ///< the energy deposited in the strip closest to the track and the neighbouring strip on each side
    Float_t qtrackx; ///< measured energy in clusters along the track in the x-view
    Float_t qtracky; ///< measured energy in clusters along the track in the y-view
    Float_t dxtrack; ///< measured energy outside the clusters along the track in the x-view
    Float_t dytrack; ///< measured energy outside the clusters along the track in the y-view
    Float_t qmean;   ///< the energy truncated mean that is the average energy deposit for the five planes with the smaller energy deposit of the whole calorimeter
    Float_t qlow;    ///< the same as "qtot" but only after the calculated electromagnetic shower maximum
    Float_t dX0l;    ///< tranversed X0 lenght
    //
    CaloTrkVar(); ///< Constructor.
    /**
     * \param trkvar Object of the class CaloTrkVar
     */
    CaloTrkVar(const CaloTrkVar &trkvar); ///< copy values from trkvar to this
    //
    void Clear(Option_t *t=""); ///< clear variables
    CaloTrkVar* GetCaloTrkVar(){return this;}; ///< returns pointer to this object
    //
    ClassDef(CaloTrkVar,2);
    //
};

/**
 * \brief Calorimeter level2 class
 *
 * This class contains level2 calorimeter variables
 *
**/
class CaloLevel2 : public TObject {
 private:
    TClonesArray *CaloTrk; ///< track related variables

 public:
    //
    // general variables
    //
    Int_t good;    ///< no errors (perr, swerr and crc are checked)
    Int_t perr[4]; ///< processing errors (one for each calorimeter section)
    Int_t swerr[4];///< DSP status word 
    Int_t crc[4];  ///< CRC errors on data
    Int_t selftrigger;///< self-trigger flag: 0 tof trigger, 2 selftrigger event, 3 selftrigger + tof trigger, 102 selftrigger event not stored in the selfdelay array  NB: selftrigger == S4 trigger || no trigger || calo trigger in the triggerconf array
    //
    // common variables (not related to tracks)
    //
    Int_t nstrip;      ///< total number of strip hit
    Int_t nsatstrip;   ///< total number of strip hit with saturated signal
    Int_t nx22;        ///< number of strip hit in the last silicon plane of the calorimeter (x view number 22)
    Int_t planemax[2]; ///< plane of maximum energy release (x and y)
    Int_t selfdelay[4][7]; ///< Delay of the selftrigger planes + coincidence for every section
    Float_t qtot;      ///< total energy detected (MIP)
    Float_t qx22;      ///< energy detected in the last silicon plane of the calorimeter (x view number 22)
    Float_t qmax;      ///< the maximum energy detected in a strip
    Float_t qq[4];     ///< the energy released in the first half of each of the four calorimeter sections
    //
    // Fit variables
    //
    Int_t npcfit[4];     ///< number of point used to perform the fit for the two views (0,1 calo fit, 2,3 selftrigger fit if any)
    Float_t varcfit[4];  ///< variance of the calorimeter fit for the two views (0,1 calo fit, 2,3 selftrigger fit if any)
    Float_t tanx[2];     ///< the tangent of the angle in the x direction as determined by the track fitted in the calorimeter (0 calo fit, 1 selftrigger fit)
    Float_t tany[2];     ///< the tangent of the angle in the x direction as determined by the track fitted in the calorimeter (0 calo fit, 1 selftrigger fit)
    Int_t fitmode[2];    ///< for x and y is 0 if the fit was performed with the "electron" algorithm, is 1 if the fit was performed with the "nuclei" algorithm
    Int_t cibar[22][2];  ///< strip traversed by the trajectory as measured by the calorimeter (calo fit)
    Float_t cbar[22][2]; ///< position in cm as measured by the calorimeter (calo fit)
    //
    // Energy variables
    //
    Float_t elen;  ///< energy in GeV assuming an electron interaction (from simulations).
    Float_t selen; ///< sigma of the energy
    //
    // track related variables: inline methods
    //
    Int_t ntrk(){return CaloTrk->GetEntries();};  ///< number of saved blocks of track-related variables
    //
    // METHODS
    //
    Float_t impx(Int_t tr);        ///< the x impact position on the first plane as determined by the track fitted in the calorimeter ( tr = 0 calo fit, tr = 1 selftrigger fit)
    Float_t impy(Int_t tr);        ///< the y impact position on the first plane as determined by the track fitted in the calorimeter ( tr = 0 calo fit, tr = 1 selftrigger fit)

    //
    void GetElectronEnergy(Float_t &energy, Float_t &sigma); ///< returns energy and sigma using qtot and assuming the particle being an electron
    //
    CaloTrkVar *GetCaloTrkVar(Int_t notrack);  ///< returns a pointer to the CaloTrkVar class containing track related variables for track number notrack
    CaloTrkVar* GetCaloStoredTrack(Int_t seqno); ///< returns pointer to the track set related to the seqno number
    //
    TClonesArray *GetTrackArray(){return CaloTrk;}; ///< returns a pointer to the track related variables array
    CaloLevel2* GetCaloLevel2(){return this;}; ///< returns pointer to this object
    //
    void GetLevel2Struct(cCaloLevel2 *l2) const;
    //
    void Clear(Option_t *t="");
    void Delete(Option_t *t=""); //ELENA
    void Set(); //ELENA
    Bool_t IsGood(Bool_t strict = false); // method to check if the event is good or not
    //
    // constructor
    //
    CaloLevel2(); ///< Constructor.
    ~CaloLevel2(){Delete();}; //ELENA
    //
    friend class CaloLevel0;
    //
    ClassDef(CaloLevel2,6);
};

#endif
