#ifndef OrbitalInfo_h
#define OrbitalInfo_h

#include <TObject.h>
#include <OrbitalInfoStruct.h>

/**
 * Class that stores position, time, inclination, magnetic field and
 * cutoff informations.
 */
class OrbitalInfo : public TObject {
 public:
  OrbitalInfo();
  OrbitalInfo* GetOrbitalInfo(){return this;}; // Elena

  UInt_t absTime; ///< Absolute Time (seconds) 
  UInt_t OBT; ///< On Board Time (ms)
  UInt_t pkt_num; ///< CPU packet number 

  Float_t lon; ///< degrees from -180 to 180
  Float_t lat; ///< degrees from -90 to 90
  Float_t alt; ///< meters asl

  // B components.
  Float_t Bnorth; ///< gauss
  Float_t Beast; ///< gauss
  Float_t Bdown; ///< gauss

  Float_t Babs; ///< abs value (guass)

  Float_t BB0; ///< B abs over the B minimum on this field line

  Float_t L; ///< McIlwain's L shell (in earth radii)

  // Dipolar magnetic coordinates (not used).
  Float_t londip; ///< degrees from -180 to 180
  Float_t latdip; ///< degrees from -90 to 90
  Float_t altdip; ///< meters

  // Corrected magnetic coordinates (not used).
  Float_t loncgm; ///< degrees from -180 to 180
  Float_t latcgm; ///< degrees from -90 to 90
  Float_t altcgm; ///< meters

  // Corrected B min magnetic coordinates (not used).
  Float_t loncbm; ///< degrees from -180 to 180
  Float_t latcbm; ///< degrees from -90 to 90
  Float_t altcbm; ///< meters

  Float_t cutoff[20];

  // Quaternions 
  Float_t q0; ///< Quaternion 0
  Float_t q1; ///< Quaternion 1
  Float_t q2; ///< Quaternion 2
  Float_t q3; ///< Quaternion 3

  // Euler angles (Resurs velocity reference frame) 
  Float_t theta; ///< Euler angle theta in the velocity reference frame (pitch)
  Float_t phi; ///< Euler angle phi in the velocity reference frame (yaw)
  Float_t etha; ///< Euler angle etha in the velocity reference frame (roll)

  /**
   * The variable mode means a character time distant between two quaternions, inside which stay every events  
   */
  Int_t mode;   // 0  - means that time different pair of quaternions exuals to 0.25 seconds in R10 mode
		//      (it mean that all quaternions in array is correct)
  		// 1  - means that we have R10 mode and use just first value of quaternions array
		// 2  - means that we have non R10 mode and use every quaternions from array.
		// 3  - means normal transition from R10 to non R10 or from non R10 to R10.
		// 4  - means that we have eliminable hole between R10 and non R10 or between non R10 and R10
		// 5  - means that we have uneliminable hole between R10 and non R10 or between non R10 and R10
		// 6  - means that we have eliminable hole inside R10 (in such keys eliminable depends from other factors also)
		// 7  - means that we have uneliminable hole inside R10
		// 8  - means that we have eliminable hole inside non R10
		// 9  - means that we have uneliminable hole inside non R10
  		
  /**
     \return Returns the B minimum along the field line.
  */
  Float_t GetB0() { return Babs/BB0; };

  /**
     \return Returns the Stormer vertical cutoff using L shell:
     14.9/L^2 (GV/c).
  */
  Float_t GetCutoffSVL() { return cutoff[0]; };

  void SetFromLevel2Struct(cOrbitalInfo *l2);
  void GetLevel2Struct(cOrbitalInfo *l2) const;

  void Clear(); // emiliano
  //
  ClassDef(OrbitalInfo, 3)
};
#endif
