/**
 * \file inc/CaloLevel2.h
 * \author Emiliano Mocchiutti
 */
#ifndef CaloLevel2_h
#define CaloLevel2_h
//
#include <TObject.h>
#include <TClonesArray.h>
#include <TArrayF.h>
//
#include <CaloStruct.h>
/**
 * \brief Calorimeter track-related variables class
 *
 * This class contains track-related variables. One set of variables is saved for any given 
 * track, including seltrigger event tracks. 
 *
 */
class CaloTrkVar : public TObject {

private:

public:
    //
    Int_t trkseqno; ///< tracker entry coming from tracker, -1 if selftrigger event.
    //
    // track related variables
    //
    Int_t ncore;     ///< SUM(j=1,2)SUM(i=1,PLmax) Nhit(i,j)*i , where Nhit(i,j) is the number of hits in a cylinder of radius 2 Rm (Moliere radius) around the track in the i-th plane (where the top plane is number 1 and the sum runs up to plane number PLmax, closest to the calculated electromagnetic shower maximum of the j-th view)
    Int_t noint;      ///< SUM(j=1,2)SUM(i=1,22) TH(i,j)*i , where TH(i,j) = 1 if the i-th plane of the j-th view has a cluster along (less than 4 mm away) the track with a deposited energy typical of a proton (order of one MIP), otherwise TH(i,j) = 0
    Int_t ncyl;      ///< the number of strip hit in a cylinder of radius 8 strips around the shower axis
    Int_t nlast;     ///< the same as "ncyl" but only for the last four planes and radius 4 strips.
    Int_t npre;      ///< the same as "ncyl" but only for the first three planes
    Int_t npresh;    ///< the same as "ncyl" but with radius 2 strips and only in the first four planes
    Int_t ntr;       ///< the same as "ncyl" but with radius 4 strips
    Int_t planetot;  ///< number of planes used to calculate the energy truncated mean "qmean"
    Int_t nlow;      ///< the same as "nstrip" but below the calculated electromagnetic shower maximum
    Int_t tibar[22][2];  ///< strip traversed by the trajectory as measured by the tracker
    Float_t tbar[22][2]; ///< position in cm as measured by the tracker
    Float_t qcore;   ///< SUM(j=1,2)SUM(i=1,PLmax) Qhit(i,j)*i , where Qhit(i,j) is the energy released (MIP) in a cylinder of radius 2 Rm (Moliere radius) around the track in the i-th plane (where the top plane is number 1 and the sum runs up to plane number PLmax, closest to the calculated electromagnetic shower maximum of the j-th view).
    Float_t qcyl;    ///< the measured energy deposited in a cylinder of radius 8 strips around the shower axis
    Float_t qlast;   ///< the same as "qcyl" but only for the last four planes and radius 4 strips.
    Float_t qpre;    ///< the same as "qcyl" but only for the first three planes
    Float_t qpresh;  ///< the same as "qcyl" but with radius 2 strips and only in the first four planes
    Float_t qtr;     ///< the same as "qcyl" but with radius 4 strips
    Float_t qtrack;  ///< the energy deposited in the strip closest to the track and the neighbouring strip on each side
    Float_t qtrackx; ///< measured energy in clusters along the track in the x-view
    Float_t qtracky; ///< measured energy in clusters along the track in the y-view
    Float_t dxtrack; ///< measured energy outside the clusters along the track in the x-view
    Float_t dytrack; ///< measured energy outside the clusters along the track in the y-view
    Float_t qmean;   ///< the energy truncated mean that is the average energy deposit for the five planes with the smaller energy deposit of the whole calorimeter
    Float_t qlow;    ///< the same as "qstrip" but below the calculated electromagnetic shower maximum
    Float_t dX0l;    ///< tranversed X0 lenght
    //
    CaloTrkVar(); ///< Constructor.
    /**
     * \param trkvar Object of the class CaloTrkVar
     */
    CaloTrkVar(const CaloTrkVar &trkvar); ///< copy values from trkvar to this
    //
    CaloTrkVar* GetCaloTrkVar(){return this;}; ///< returns pointer to this object
    //
    ClassDef(CaloTrkVar,1);
    //
};

/**
 * \brief Calorimeter level2 class
 *
 * This class contains level2 calorimeter variables
 *
**/
class CaloLevel2 : public TObject {
 private:
    TClonesArray *CaloTrk; ///< track related variables

 public:
    //
    // general variables
    //
    Int_t good;    ///< no errors (perr, swerr and crc are checked)
    Int_t perr[4]; ///< processing errors (one for each calorimeter section)
    Int_t swerr[4];///< DSP status word 
    Int_t crc[4];  ///< CRC errors on data
    Int_t selftrigger;///< self-trigger flag (1 selftrigger event, 0 normal event)
    //
    // common variables (not related to tracks)
    //
    Int_t nstrip;      ///< total number of strip hit
    Int_t nx22;        ///< number of strip hit in the last silicon plane of the calorimeter (x view number 22)
    Int_t planemax[2]; ///< plane of maximum energy release (x and y)
    Float_t qtot;      ///< total energy detected (MIP)
    Float_t qx22;      ///< energy detected in the last silicon plane of the calorimeter (x view number 22)
    Float_t qmax;      ///< the maximum energy detected in a strip
    Float_t qq[4];     ///< the energy released in the first half of each of the four calorimeter sections
    //
    // Fit variables
    //
    Int_t npcfit[2];     ///< number of point used to perform the fit for the two views
    Int_t cibar[22][2];  ///< strip traversed by the trajectory as measured by the calorimeter
    Float_t cbar[22][2]; ///< position in cm as measured by the calorimeter
    Float_t impx;        ///< the x impact position on the first plane as determined by the track fitted in the calorimeter
    Float_t impy;        ///< the y impact position on the first plane as determined by the track fitted in the calorimeter
    Float_t tanx;        ///< the tangent of the angle in the x direction as determined by the track fitted in the calorimeter
    Float_t tany;        ///< the tangent of the angle in the x direction as determined by the track fitted in the calorimeter
    Float_t varcfit[2];  ///< variance of the calorimeter fit for the two views
    //
    // Energy variables
    //
    Float_t elen;  ///< energy in GeV assuming an electron interaction (from simulations).
    Float_t selen; ///< sigma of the energy
    //
    // track related variables: inline methods
    //
    Int_t ntrk()                                       {return CaloTrk->GetEntries();};  ///< number of saved blocks of track-related variables
    Int_t trkseqno(Int_t entry)                        {return (((CaloTrkVar *)CaloTrk->At(entry))->trkseqno);}; ///< extract trkseqno
    Int_t ncore(Int_t entry)                           {return (((CaloTrkVar *)CaloTrk->At(entry))->ncore);}; ///< extract ncore
    Int_t noint(Int_t entry)                           {return (((CaloTrkVar *)CaloTrk->At(entry))->noint);}; ///< extract noint
    Int_t ncyl(Int_t entry)                            {return (((CaloTrkVar *)CaloTrk->At(entry))->ncyl);}; ///< extract ncyl
    Int_t nlast(Int_t entry)                           {return (((CaloTrkVar *)CaloTrk->At(entry))->nlast);}; ///< extract nlast
    Int_t npre(Int_t entry)                            {return (((CaloTrkVar *)CaloTrk->At(entry))->npre);}; ///< extract npre
    Int_t npresh(Int_t entry)                          {return (((CaloTrkVar *)CaloTrk->At(entry))->npresh);}; ///< extract npresh
    Int_t ntr(Int_t entry)                             {return (((CaloTrkVar *)CaloTrk->At(entry))->ntr);}; ///< extract ntr
    Int_t nlow(Int_t entry)                            {return (((CaloTrkVar *)CaloTrk->At(entry))->nlow);}; ///< extract nlow
    Int_t planetot(Int_t entry)                        {return (((CaloTrkVar *)CaloTrk->At(entry))->planetot);}; ///< extract planetot
    Int_t tibar(Int_t entry, Int_t plane, Int_t view)  {return (((CaloTrkVar *)CaloTrk->At(entry))->tibar[plane][view]);}; ///< extract tibar
    Float_t tbar(Int_t entry, Int_t plane, Int_t view) {return (((CaloTrkVar *)CaloTrk->At(entry))->tbar[plane][view]);}; ///< extract tbar
    Float_t qcore(Int_t entry)                         {return (((CaloTrkVar *)CaloTrk->At(entry))->qcore);}; ///< extract qcore
    Float_t qcyl(Int_t entry)                          {return (((CaloTrkVar *)CaloTrk->At(entry))->qcyl);}; ///< extract qcyl
    Float_t qlast(Int_t entry)                         {return (((CaloTrkVar *)CaloTrk->At(entry))->qlast);}; ///< extract qlast
    Float_t qpre(Int_t entry)                          {return (((CaloTrkVar *)CaloTrk->At(entry))->qpre);}; ///< extract qpre
    Float_t qpresh(Int_t entry)                        {return (((CaloTrkVar *)CaloTrk->At(entry))->qpresh);}; ///< extract qpresh
    Float_t qtr(Int_t entry)                           {return (((CaloTrkVar *)CaloTrk->At(entry))->qtr);}; ///< extract qtr
    Float_t qtrack(Int_t entry)                        {return (((CaloTrkVar *)CaloTrk->At(entry))->qtrack);}; ///< extract qtrack
    Float_t qtrackx(Int_t entry)                       {return (((CaloTrkVar *)CaloTrk->At(entry))->qtrackx);}; ///< extract qtrackx
    Float_t qtracky(Int_t entry)                       {return (((CaloTrkVar *)CaloTrk->At(entry))->qtracky);}; ///< extract qtracky
    Float_t dxtrack(Int_t entry)                       {return (((CaloTrkVar *)CaloTrk->At(entry))->dxtrack);}; ///< extract dxtrack
    Float_t dytrack(Int_t entry)                       {return (((CaloTrkVar *)CaloTrk->At(entry))->dytrack);}; ///< extract dytrack
    Float_t qmean(Int_t entry)                         {return (((CaloTrkVar *)CaloTrk->At(entry))->qmean);}; ///< extract qmean
    Float_t qlow(Int_t entry)                          {return (((CaloTrkVar *)CaloTrk->At(entry))->qlow);}; ///< extract qlow
    Float_t dX0l(Int_t entry)                          {return (((CaloTrkVar *)CaloTrk->At(entry))->dX0l);}; ///< extract dX0l
    //
    // Number of strip with energy > emip and their value coded with view plane and strip number:
    // view x[y] plane PP strip SS with energy mmmm.iip = +[-] ( PP*10^6 + SS*10^4 + mmmm.iip )
    //
    TArrayF estrip; ///< MIP values for each strip with energy > emin coded with view plane and strip number; view x[y] plane PP strip SS with energy mmmm.iip = +[-] ( PP*10^6 + SS*10^4 + mmmm.iip )
    //
    // METHODS
    //
    void GetElectronEnergy(Float_t &energy, Float_t &sigma); ///< returns energy and sigma using qtot and assuming the particle being an electron
    Float_t GetEstrip(Int_t view, Int_t plane, Int_t strip); ///< returns saved MIP value for the indicated strip
    Float_t DecodeEstrip(Int_t entry, Int_t &view, Int_t &plane, Int_t &strip); ///< returns saved MIP value for the entry number "entry" of the TArrayF.
    CaloTrkVar *GetCaloTrkVar(Int_t notrack);  ///< returns a pointer to the CaloTrkVar class containing track related variables
    //
    TClonesArray *GetTrackArray(){return CaloTrk;};
    CaloLevel2* GetCaloLevel2(){return this;}; ///< returns pointer to this object
    //
    void GetLevel2Struct(cCaloLevel2 *l2) const;
    void Clear();
    //
    // constructor
    //
    CaloLevel2();
    //
    friend class CaloProcessing;
    //
    ClassDef(CaloLevel2,1);
};

#endif
